DROP VIEW IF EXISTS sequence_view;
DROP VIEW IF EXISTS structure_sequence_view;
CREATE VIEW structure_sequence_view AS
WITH cte AS (
  SELECT
    assembly.structure_id AS structure_id,
    sequence_id, array_agg(residue.id) AS residue_ids,
    array_agg(assembly.id) AS assembly_ids,
    structure.wwpdb,
    structure.afdb,
    structure.method,
    structure.resolution
  FROM sequence_residue_mapping
     JOIN residue ON sequence_residue_mapping.residue_id = residue.id
     JOIN chain ON residue.chain_id = chain.id
     JOIN assembly ON chain.assembly_id = assembly.id
     JOIN structure ON assembly.structure_id = structure.id
  GROUP BY assembly.structure_id, sequence_id, structure.wwpdb, structure.afdb, structure.method, structure.resolution
)
SELECT
  cte.structure_id as id,
  sequence_id AS _sequence_id,
  cte.wwpdb,
  cte.afdb,
  cte.method,
  cte.resolution,
  -- get residues
  COALESCE ((
    SELECT jsonb_agg(jsonb_build_object(
      'id', id,
      'chainName', (SELECT name FROM chain WHERE id = chain_id),
      'structPosition', struct_position,
      'insertionCode', insertion_code,
      'aminoAcid', amino_acid,
      'seqPosition', (SELECT position FROM sequence_residue_mapping WHERE residue_id = id),
      'asa', asa,
      'bFactor', b_factor,
      'secondaryStructure', secondary_structure,
      'inTunnel', NULL,
      'inPocket', NULL
    ))
    FROM residue WHERE residue.id = ANY(residue_ids)
  ), '[]'::jsonb) AS residues,
  -- get tunnels
  COALESCE((
     SELECT jsonb_agg(DISTINCT jsonb_build_object(
      'number', void.number,
      'spheres', '[]'::jsonb,
      'residues', COALESCE((
        SELECT jsonb_agg(residue_id) FROM void_residue WHERE void_id = tunnel.id
       ), '[]'::jsonb),
      'curvature', tunnel.curvature,
      'distanceToSurface', tunnel.distance_to_surface,
      'length', tunnel.length,
      'throughput', tunnel.throughput
    ))
    FROM void
    JOIN tunnel ON void.id = tunnel.id
    WHERE void.assembly_id = ANY(assembly_ids)
  ), '[]'::jsonb) AS tunnels,
  -- get pockets
  COALESCE((
    SELECT jsonb_agg(DISTINCT jsonb_build_object(
      'number', void.number,
      'spheres', '[]'::jsonb,
      'residues', COALESCE((
        SELECT jsonb_agg(residue_id) FROM void_residue WHERE void_id = pocket.id
      ), '[]'::jsonb),
      'druggability', pocket.druggability,
      'relevance', pocket.relevance,
      'volume', pocket.volume,
      'centerX', pocket.center_x,
      'centerY', pocket.center_y,
      'centerZ', pocket.center_z
    ))
    FROM void
    JOIN pocket ON void.id = pocket.id
    WHERE void.assembly_id = ANY(assembly_ids)
  ), '[]'::jsonb) AS pockets
FROM cte;


----------------- SEQUENCE VIEW -------------------
CREATE VIEW sequence_view AS
SELECT
SEQ.id,
SEQ.sequence,
length(SEQ.sequence) AS length,
-- get protein links
COALESCE((
  SELECT jsonb_agg(DISTINCT jsonb_build_object(
    'protein',
    (
    SELECT DISTINCT jsonb_build_object(
      'id', prot.id,
      'name', prot.name,
      'organism', prot.organism,
      'references', COALESCE((
        SELECT jsonb_agg(DISTINCT jsonb_build_object(
          'type', ref.type,
          'accession', ref.accession,
          'name', ref.name
        ))
        FROM protein_reference ref WHERE ref.protein_id = prot.id
      ), '[]'::jsonb)
    )
    FROM protein prot WHERE prot.id = protseq.protein_id LIMIT 1
    ),
  'isoform', protseq.isoform
  ))
  FROM protein_sequence protseq WHERE protseq.sequence_id = SEQ.id LIMIT 1
  ), '[]'::jsonb
) AS "proteinLinks",
-- get features
COALESCE((
  SELECT jsonb_agg(DISTINCT jsonb_build_object(
    'id', sf.id,
    'type', sf.type,
    'position', sf.position,
    'positionRange', CASE WHEN sf.position_range IS NOT NULL THEN jsonb_build_array(
      CASE WHEN lower_inc(sf.position_range) THEN lower(sf.position_range) ELSE lower(sf.position_range) + 1 END,
      CASE WHEN upper_inc(sf.position_range) THEN upper(sf.position_range) ELSE upper(sf.position_range) - 1 END
    ) END,
    'positionArray', sf.position_array,
    'numValue', sf.num_value,
    'strValue', sf.str_value
  ))
  FROM sequence_feature sf WHERE sf.sequence_id = SEQ.id), '[]'::jsonb
) AS features,
-- get structures
COALESCE((
  SELECT jsonb_agg(to_jsonb(structure_sequence_view.*))
  FROM structure_sequence_view
  WHERE structure_sequence_view._sequence_id = SEQ.id
), '[]'::jsonb) AS structures,
-- get experiments
COALESCE((
  SELECT jsonb_agg(DISTINCT jsonb_build_object(
    'id', ex.id,
    'publication', (
      SELECT DISTINCT jsonb_build_object(
        'id', pub.id,
        'type', pub.type,
        'title', pub.title,
        'journal', pub.journal,
        'year', pub.year,
        'doi', pub.doi,
        'pmid', pub.pmid,
        'url', pub.url,
        'authors', COALESCE((
          SELECT jsonb_agg(DISTINCT jsonb_build_object(
            'id', auth.id,
            'orcid', auth.orcid,
            'name', auth.name
          ))
          FROM author auth
          WHERE auth.id IN (
            SELECT author_id FROM publication_author WHERE publication_id = pub.id
          )
        ), '[]'::jsonb)
      )
      FROM publication pub WHERE pub.id = ex.publication_id LIMIT 1
    ),
    'dataset', (
      SELECT dataset.name FROM dataset WHERE dataset.id = ex.dataset_id
    ),
    'measurements', COALESCE((
      SELECT jsonb_agg(DISTINCT jsonb_build_object(
        'id', mes.id,
        'type', mes.type,
        'numValue', mes.num_value,
        'strValue', mes.str_value,
        'datasets', COALESCE((
          SELECT jsonb_agg(DISTINCT dataset.name)
          FROM dataset_measurement dm
          JOIN dataset ON dataset.id = dm.dataset_id
          WHERE dm.measurement_id = mes.id
        ), '[]'::jsonb),
        'references', COALESCE((
          SELECT jsonb_agg(DISTINCT jsonb_build_object(
            'type', ref.type,
            'accession', ref.accession
          ))
          FROM measurement_reference ref WHERE ref.measurement_id = mes.id
        ), '[]'::jsonb)
      ))
      FROM measurement mes WHERE mes.experiment_id = ex.id
    ), '[]'::jsonb),
    'annotations', COALESCE((
      SELECT jsonb_agg(DISTINCT jsonb_build_object(
        'type', ann.type,
        'numValue', ann.num_value,
        'strValue', ann.str_value
      ))
      FROM experiment_annotation ann WHERE ann.experiment_id = ex.id
    ), '[]'::jsonb)
  ))
  FROM experiment ex
  WHERE
    ex.id IN (SELECT ms.experiment_id FROM measurement ms WHERE ms.sequence_id = SEQ.id)
    AND ex.active IS TRUE
  ), '[]'::jsonb) AS experiments
FROM sequence SEQ;
